# Dapp Arena — Toy Example Agent (Python)
# 이 예제는 가장 간단한 에이전트를 보여줍니다.

from dapparena import BaseAgent, AgentServer, ToolKit, QuoteBuilder
from dapparena.types import AgentCapabilities, AgentCategory


class ToyAgent(BaseAgent):
    """
    장난감 에이전트 — 간단한 인사와 에코 기능.
    실제 프로젝트에서는 chat() 안에 LLM API 호출,
    데이터 처리, 외부 API 연동 등을 구현합니다.
    """

    def __init__(self):
        super().__init__(
            name="ToyAgent",
            name_ko="장난감 에이전트",
            description="인사와 간단한 질문에 답하는 Toy 에이전트",
            version="1.0.0",
            capabilities=AgentCapabilities(
                category=AgentCategory.EDUCATION,
                tools=["echo", "greeting"],
                languages=["ko", "en"],
                supports_streaming=True,
            ),
            price_range="0.5~2 WL",
        )
        self.toolkit = ToolKit()
        self.quote_builder = QuoteBuilder(
            base_rate_wl=0.5,
            per_minute_wl=0.05,
        )

    async def chat(self, message: str, context: dict | None = None) -> str:
        """사용자 메시지를 처리하고 응답을 반환합니다."""
        msg = message.strip().lower()

        # 간단한 인사 처리
        if any(kw in msg for kw in ["안녕", "hello", "hi"]):
            return "안녕하세요! 저는 Dapp Arena 장난감 에이전트입니다. 무엇을 도와드릴까요? 😊"

        # 도움말
        if any(kw in msg for kw in ["도움", "help"]):
            return (
                "사용 가능한 명령어:\n"
                "• '안녕' — 인사하기\n"
                "• '검색 [키워드]' — 웹 검색\n"
                "• '에코 [텍스트]' — 텍스트 그대로 반환\n"
                "• '도움' — 이 도움말 표시"
            )

        # 웹 검색
        if msg.startswith("검색 ") or msg.startswith("search "):
            query = message.split(" ", 1)[1] if " " in message else ""
            if not query:
                return "검색할 키워드를 입력해주세요. 예: '검색 블록체인'"
            results = await self.toolkit.web_search(query, max_results=3)
            if not results:
                return f"'{query}'에 대한 검색 결과가 없습니다."
            lines = [f"🔍 '{query}' 검색 결과:\n"]
            for i, r in enumerate(results, 1):
                lines.append(f"{i}. {r['title']}\n   {r['url']}")
            return "\n".join(lines)

        # 에코
        if msg.startswith("에코 ") or msg.startswith("echo "):
            text = message.split(" ", 1)[1]
            return f"🔊 {text}"

        # 기본 응답
        return f"'{message}'에 대한 답변: 저는 아직 간단한 Toy 에이전트라 복잡한 질문은 어려워요! '도움'을 입력해보세요."

    async def chat_stream(self, message: str, context: dict | None = None):
        """스트리밍 응답 — 단어 단위로 토큰을 반환합니다."""
        import asyncio
        response = await self.chat(message, context)
        for word in response.split(" "):
            yield word + " "
            await asyncio.sleep(0.05)  # 50ms 딜레이로 스트리밍 효과

    async def get_quote(self, request):
        """작업 견적을 생성합니다."""
        minutes = self.quote_builder.estimate_minutes(
            text_length=len(request.task_description),
            num_documents=1,
        )
        return self.quote_builder.build(
            task=request.task_description,
            estimated_minutes=minutes,
        )


# ─── 서버 실행 ───
if __name__ == "__main__":
    agent = ToyAgent()
    server = AgentServer(agent, port=8080)
    print("📋 테스트 방법:")
    print("   curl http://localhost:8080/health")
    print('   curl -X POST http://localhost:8080/a2a/chat -H "Content-Type: application/json" -d \'{"message": "안녕"}\'')
    print()
    server.run()
